from django.db import models

# Extends base user model
from django.contrib.auth.models import AbstractUser
from django.db.models.fields import proxy
from django.db.models.signals import pre_save, post_save
from django.dispatch import receiver
from rest_framework.authtoken.models import Token
from django.utils import timezone
from django_resized import ResizedImageField
from ckeditor_uploader.fields import RichTextUploadingField
from taggit.managers import TaggableManager
from django.utils.text import slugify
from api.utils.blog_utils import count_words, read_time
import uuid
from .course_models import Courses
from django.core.validators import FileExtensionValidator
from django import forms
from django_lifecycle import LifecycleModel, hook, AFTER_DELETE, AFTER_SAVE
from django.core.cache import cache



class User(AbstractUser):
    first_name = models.CharField(max_length=100)
    middle_name = models.CharField(max_length=100, blank=True)
    last_name = models.CharField(max_length=100)
    phone_number = models.CharField(max_length=30)
    email = models.EmailField(unique=True)
    password = models.CharField(max_length=100)
    date_joined = models.DateTimeField(default=timezone.now, blank=True)
    username = models.CharField(max_length=100, blank=True)
    is_verified = models.BooleanField(default=False)
    image = models.ImageField(blank=True, null=True, upload_to="user_image")

    REQUIRED_FIELDS = [
        "first_name",
        "last_name",
        "phone_number",
        "password",
        "username",
    ]

    USERNAME_FIELD = "email"

    def get_username(self):
        return self.email

    def get_full_name(self) -> str:
        return super().get_full_name()

    def __str__(self):
        return str(self.email)


class review(models.Model):
    user = models.ForeignKey(User, on_delete=models.CASCADE)
    course = models.ForeignKey(Courses, on_delete=models.CASCADE)
    rate_choices = (("1", "1"), ("2", "2"), ("3", "3"), ("4", "4"), ("5", "5"))
    rate = models.CharField(choices=rate_choices, max_length=5)

    def __str__(self):
        return self.rate


class course_batch(models.Model):
    course = models.ForeignKey(
        Courses, on_delete=models.CASCADE, blank=True, null=True)
    batch_number = models.CharField(max_length=10)
    # This is to show number of submission recieved so far
    submission_number = models.PositiveIntegerField(default=0)
    next_batch_date = models.DateField(blank=True, null=True)
    next_batch_time = models.CharField(blank=True, null=True, max_length=30)
    active = models.BooleanField(default=False)

    def __str__(self):
        return f"{self.course} {self.batch_number}"

    class Meta:
        verbose_name_plural = "course_batches"


class course_registred_user(models.Model):
    user = models.ForeignKey(
        User, on_delete=models.CASCADE, blank=True, related_name="user", null=True
    )
    course = models.ForeignKey(
        Courses, on_delete=models.CASCADE, blank=True, null=True)
    first_name = models.CharField(max_length=100)
    middle_name = models.CharField(max_length=100, blank=True)
    last_name = models.CharField(max_length=100)
    company_institution = models.CharField(max_length=500)
    phone_number = models.CharField(max_length=30, blank=True)
    email = models.EmailField(unique=False)
    zip_number = models.CharField(max_length=100, blank=True)
    professional_academic_field = models.CharField(max_length=500)
    address = models.CharField(max_length=500, blank=True)
    country = models.CharField(max_length=500)
    number_of_slots = models.PositiveIntegerField(default=1)
    reference_id = models.UUIDField(blank=True)
    payment_reference_id = models.UUIDField(blank=True, null=True)
    paid = models.BooleanField(default=False)
    pay_later = models.BooleanField(default=True)
    batch_no = models.CharField(max_length=5, blank=True, null=True)
    registration_id = models.CharField(max_length=30, blank=True)
    date_registered = models.DateTimeField(default=timezone.now, blank=True)
    completed = models.BooleanField(default=False)
    postponed = models.BooleanField(default=False)
    archived = models.BooleanField(default=False)
    start_date = models.DateField(blank=True, null=True)
    end_date = models.DateField(blank=True, null=True)
    certificate = models.BooleanField(default=False)
    attendance = models.PositiveIntegerField(default=0)
    # email_sent =

    def __str__(self):
        return f"{self.registration_id}"

    def save(self, *args, **kwargs):
        # self.get_batch_no()
        super(course_registred_user, self).save(*args, **kwargs)

    

    class Meta:
        verbose_name_plural = "Course Registerd Participants"
        ordering = ["user", "course"]


class course_registered_user_proxy(course_registred_user):
    class Meta:
        proxy = True


class extra_participant(models.Model):
    registra = models.ForeignKey(
        course_registred_user, on_delete=models.CASCADE)
    first_name = models.CharField(max_length=100)
    middle_name = models.CharField(max_length=100, blank=True)
    last_name = models.CharField(max_length=100)
    email = models.EmailField(unique=False)
    course = models.ForeignKey(
        Courses, on_delete=models.CASCADE, null=True, blank=True)
    # reg_id = models.CharField(max_length=100, blank=True)

    def __str__(self):
        return self.email

    class Meta:
        verbose_name_plural = "Extra Participant"


class project_batch(models.Model):
    batch_number = models.CharField(max_length=10)
    # This is to show number of submission recieved so far
    submission_number = models.PositiveIntegerField(default=0)
    number_of_project_excuted = models.PositiveIntegerField(default=0)

    def __str__(self):
        return f"Project batch {self.batch_number}"

    class Meta:
        verbose_name_plural = "Project Batches"


class project(models.Model):
    execution_option = (
        ("Full execution", "Full execution"),
        ("Project assistance", "Project assistance"),
    )
    # project = models.ForeignKey("project_submission", on_delete=models.CASCADE, blank=True)
    project_title = models.CharField(max_length=400, blank=True)
    project_full_name = models.CharField(max_length=600, blank=True)
    country = models.CharField(max_length=500, blank=True)
    project_nature = models.CharField(choices=execution_option, max_length=200)
    project_type = models.TextField()
    project_scope = models.CharField(max_length=300, blank=True)
    executed = models.BooleanField(default=False)
    image = models.ImageField(upload_to="projects", blank=True)
    detail_image = models.ImageField(upload_to="projects", blank=True)
    meta = models.TextField(blank=True)
    project_duration = models.CharField(max_length=100, blank=True)
    project_completion_date = models.CharField(
        max_length=20, blank=True, null=True)
    prid = models.CharField(max_length=100, blank=True, unique=True)

    def __str__(self):
        return f"{self.project_title} - {self.prid}"
    # def save(self, *args, **kwargs):
    #
    #     super(project, self).save(*args, **kwargs)

    @hook(AFTER_SAVE)
    @hook(AFTER_DELETE)
    def invalidate_cache(self):
        cache.delete("projects")
        cache.delete("project_list")


class project_submission(models.Model):
    user = models.ForeignKey(
        User, on_delete=models.SET_NULL, blank=True, null=True)
    project_title = models.CharField(max_length=400, blank=True)
    first_name = models.CharField(max_length=200)
    middle_name = models.CharField(max_length=200, blank=True)
    last_name = models.CharField(max_length=250)
    company_university = models.CharField(max_length=300)
    email = models.EmailField()
    phone_number = models.CharField(max_length=30)
    zip_number = models.CharField(max_length=10, blank=True)
    professional_academic_field = models.CharField(max_length=500)
    address = models.CharField(max_length=500, blank=True)
    country = models.CharField(max_length=500, blank=True)
    project_file = models.FileField(
        blank=True,
        null=True,
        upload_to="project_file"
    )
    comment = models.TextField(blank=True)
    project_id = models.CharField(max_length=100, blank=True)
    # batch_no = models.ForeignKey(project_batch, on_delete = models.CASCADE, blank = True)
    project_nature = models.CharField(
        choices=project.execution_option, max_length=200)
    project_type = models.TextField(blank=True)
    date_submitted = models.DateField(
        default=timezone.now, blank=True, null=True)

    def __str__(self):
        return self.project_title

    class Meta:
        verbose_name_plural = "Project Submissions"


class payment(models.Model):
    date = models.DateField(default=timezone.now)
    user = models.ForeignKey(User, on_delete=models.CASCADE)
    reference_number = models.UUIDField()
    amount = models.PositiveIntegerField()
    status = models.CharField(max_length=50)
    payment_for = models.CharField(max_length=20, blank=True)

    def __str__(self):
        return self.user.username


class paypal_access(models.Model):
    token = models.TextField()
    expires_in = models.PositiveIntegerField()
    created_at = models.DateTimeField(default=timezone.now)

    def __str__(self):
        return self.token

    class Meta:
        verbose_name_plural = "Paypal Access"


class IpModel(models.Model):
    ip = models.CharField(max_length=150)

    def __str__(self):
        return self.ip


class blogs(models.Model):
    status_choice = (("Drafted", "Drafted"), ("Published", "Published"))
    title = models.TextField(blank=True)
    date_published = models.DateField(default=timezone.now, blank=True)
    date_created = models.DateField(default=timezone.now, blank=True)
    author = models.ManyToManyField("Author", blank=True)
    image = ResizedImageField(
        size=[500, 500], upload_to="article_pics", force_format="PNG", blank=True, null=True
    )
    content = RichTextUploadingField(blank=True)
    full_content = RichTextUploadingField(blank=True)
    # slug = models.SlugField(blank=True)
    tags = TaggableManager(blank=True)
    status = models.CharField(
        max_length=20, choices=status_choice, default="DRAFT")
    views = models.PositiveIntegerField(default=0)
    count_words = models.CharField(max_length=500, default=0, blank=True)
    read_time = models.CharField(max_length=500, default=0, blank=True)
    deleted = models.BooleanField(default=False, blank=True)
    bid = models.CharField(max_length=10, blank=True)
    meta = models.TextField(blank=True)
    last_updated = models.DateField(default=timezone.now)

    class Meta:
        ordering = ("-date_published",)
        verbose_name_plural = "blogs"

    def __str__(self):
        return self.title

    def save(self, *args, **kwargs):
        # self.slug = slugify(self.title, allow_unicode=True)
        self.count_words = count_words(self.content)
        self.read_time = read_time(self.content)
        self.last_updated = timezone.now()
        super(blogs, self).save(*args, **kwargs)


class Author(models.Model):
    name = models.CharField(max_length=200, blank=True)
    image = ResizedImageField(size=[300, 300], upload_to="author", blank=True)
    about = models.TextField(blank=True)

    def __str__(self):
        return self.name


class comment(models.Model):
    name = models.ForeignKey(User, on_delete=models.CASCADE, blank=True)
    content = models.TextField()
    date_commented = models.DateTimeField(default=timezone.now, blank=True)
    article = models.ForeignKey(blogs, on_delete=models.CASCADE, blank=True)
    reply = models.TextField(blank=True)

    def __str__(self):
        return self.content


# Password Reset code
class passwordresetcode(models.Model):
    code = models.CharField(max_length=200)
    created_at = models.DateTimeField(default=timezone.now)
    user = models.ForeignKey(User, on_delete=models.CASCADE)
    checked = models.BooleanField(default=False)

    def __str__(self):
        return self.code

    class Meta:
        verbose_name_plural = "Passsword Reset"


# Password Reset code
class emailconfirmationcode(models.Model):
    code = models.CharField(max_length=200)
    created_at = models.DateTimeField(default=timezone.now)
    user = models.ForeignKey(User, on_delete=models.CASCADE)

    def __str__(self):
        return self.code

    class Meta:
        verbose_name_plural = "Email Confirmation"


class Subscription(models.Model):
    user = models.ForeignKey(User, on_delete=models.CASCADE)
    subscribed = models.BooleanField(default=False)
    date_created = models.DateTimeField(default=timezone.now)
    last_sub_date = models.DateTimeField(blank=True, null=True, help_text="This is to keep track of when a user subscribed")
    max_number_of_days = models.PositiveIntegerField(default=31)
    def __str__(self) -> str:
        return f"{self.user.email}"

    class Meta:
        verbose_name_plural = "Subscriptions"
        # unique_together = ['user', 'label']


class saved_blogs(models.Model):
    blog = models.ManyToManyField(blogs)
    user = models.ForeignKey(User, on_delete=models.CASCADE, blank=True)
    date_saved = models.DateTimeField(default=timezone.now)

    def __str__(self):
        return self.user.email

    class Meta:
        verbose_name_plural = "Saved Blogs"


class Project_files(models.Model):
    project_for = models.ForeignKey(project, on_delete=models.CASCADE)
    # project_name = models.CharField(max_length=500)
    project_report = models.FileField(
        upload_to="project reports",
        validators=[FileExtensionValidator(allowed_extensions=["pdf", "zip"])]
    )
    useful_links = models.URLField(blank=True, null=True)
    date_added = models.DateField(default=timezone.now)

    def __str__(self) -> str:
        return self.project_for.project_title

    class Meta:
        verbose_name_plural = "Project Files"


class course_certificate(models.Model):
    reg_obj = models.ForeignKey(
        course_registred_user, on_delete=models.SET_NULL, null=True
    )
    certificate_file = models.FileField(
        upload_to="Certificates",
        validators=[
            FileExtensionValidator(
                allowed_extensions=["pdf", "zip", "png", "jpeg", "jpg"]
            )
        ]
    )
    date_added = models.DateField(default=timezone.now)

    def __str__(self):
        return f"{self.reg_obj.user.email} - {self.reg_obj.course}"

    class Meta:
        ordering = ["reg_obj"]


class Ads(models.Model):
    code = models.TextField()
    page = models.CharField(max_length=50, null=True, default="blog")

    def __str__(self):
        return f"{self.page} ads - {self.pk}"
# Signal functions


# post save is used because we want to ensure token is only generated for existing user
@receiver(post_save, sender=User)
def create_auth_user(sender, instance=None, created=False, **kwargs):
    if created:
        Token.objects.create(user=instance)
        # Subscription.objects.create(user=instance, label = 'project')
        # Subscription.objects.create(user=instance, label = 'learn')


@receiver(post_save, sender=project)
def create_auth_user(sender, instance=None, created=False, **kwargs):
    if created:
        instance.prid = str(instance.project_title[:5]) + str(instance.pk)
        instance.save()
