from rest_framework.authtoken.models import Token

from datetime import timedelta
from django.utils import timezone
from django.conf import settings

from rest_framework.authentication import TokenAuthentication
from rest_framework.exceptions import AuthenticationFailed
from rest_framework.response import Response
from django.http.response import JsonResponse


# this return left time
def expires_in(token):
    time_elapsed = timezone.now() - token.created
    left_time = timedelta(
        seconds=settings.TOKEN_EXPIRED_AFTER_SECONDS) - time_elapsed
    return left_time

# token checker if token expired or not
def is_token_expired(token):
    return expires_in(token) < timedelta(seconds=0)


# If token is expired then it will be removed and new one with different key will be created
def token_expire_handler(token, status=None):
    # Check if the token has expired
    is_expired = is_token_expired(token)
    if is_expired:
        # delete the expired token from the database
        token.delete()
        if status == "login":
            # The login is to ensure new token is created solely through the login function call
            # user can be fetched from the present local token in the function
            token = Token.objects.create(user=token.user)
            return is_expired, token

        return is_expired, token
    return is_expired, token


# DEFAULT_AUTHENTICATION_CLASSES
class ExpiringTokenAuthentication(TokenAuthentication):
    """
    If token is expired then it will be removed
    and new one with different key will be created
    """

    def authenticate_credentials(self, key):
        try:
            token = Token.objects.get(key=key)
        except Token.DoesNotExist:
            # raise an error of invalid for token that does not exist
            raise AuthenticationFailed({"validity": "invalid"})

        # This is to take care of users who are yet to activate their account
        if not token.user.is_verified:
            raise AuthenticationFailed("Please confirm your email first")

        # cheack for the validity of the token if it has been created within last an hour
        is_expired, token = token_expire_handler(token)
        if is_expired:
            # redirect the user to the login page for new login which enable new token to be returned to the fronend
            raise AuthenticationFailed({"validity": "invalid"})

        return (token.user, token)
